<?php
function generate_credit_card($bin, $amount, $month = null, $year = null, $cvv = null) {
    $cards = [];
    $numeric_bin = preg_replace('/[xX]/', '', $bin);
    $bin_info = get_bin_info(substr($numeric_bin, 0, 6));
    $card_length = 16;
    if ($bin_info && $bin_info['Status'] === 'SUCCESS') {
        if (strtoupper($bin_info['Scheme']) === 'AMERICAN EXPRESS') {
            $card_length = 15;
        }
    }

    $x_count = substr_count(strtolower($bin), 'x');
    $fixed_digits = strlen($bin) - $x_count;

    if ($fixed_digits >= $card_length || ($fixed_digits + $x_count) > $card_length) {
        return [];
    }

    for ($i = 0; $i < $amount; $i++) {
        while (true) {
            $card_body = '';
            for ($j = 0; $j < strlen($bin); $j++) {
                $card_body .= (strtolower($bin[$j]) === 'x') ? rand(0, 9) : $bin[$j];
            }

            $remaining_digits = ($card_length - 1) - strlen($card_body);
            for ($j = 0; $j < $remaining_digits; $j++) {
                $card_body .= rand(0, 9);
            }

            if (strlen($card_body) !== ($card_length - 1)) {
                continue;
            }

            $check_digit = calculate_luhn_check_digit($card_body);
            $card_number = $card_body . $check_digit;

            if (luhn_algorithm($card_number)) {
                $card_month = $month ?? sprintf("%02d", rand(1, 12));
                $card_year = $year ?? rand(2024, 2029);
                $card_cvv = $cvv ?? sprintf("%03d", ($card_length == 15 ? rand(0, 9999) : rand(0, 999)));
                $cards[] = "$card_number|$card_month|$card_year|$card_cvv";
                break;
            }
        }
    }
    return $cards;
}

function parse_input($text) {
    $pattern = '/^(\d{6,15}[xX]{0,12})(?:[|;:\/](\d{2}))?(?:[|;:\/](\d{2,4}))?(?:[|;:\/](\d{3,4}))?(?:\s+(\d{1,4}))?$/';
    preg_match($pattern, trim($text), $matches);
    if (empty($matches)) return [null, null, null, null, null];

    $bin = $matches[1] ?? null;
    $month = isset($matches[2]) ? sprintf("%02d", $matches[2]) : null;
    $year = isset($matches[3]) ? (strlen($matches[3]) == 2 ? "20{$matches[3]}" : $matches[3]) : null;
    $cvv = isset($matches[4]) ? $matches[4] : null;
    $amount = isset($matches[5]) ? (int)$matches[5] : 10;

    if (!$bin) return [null, null, null, null, null];

    $numeric_bin = preg_replace('/[xX]/', '', $bin);

    if (strlen($numeric_bin) < 6) {
        return [null, null, null, null, null];
    }

    $bin_info = get_bin_info(substr($numeric_bin, 0, 6));
    $card_length = 16;
    if ($bin_info && $bin_info['Status'] === 'SUCCESS' && strtoupper($bin_info['Scheme']) === 'AMERICAN EXPRESS') {
        $card_length = 15;
    }

    $x_count = substr_count(strtolower($bin), 'x');
    $fixed_digits = strlen($bin) - $x_count;

    if ($fixed_digits >= $card_length || ($fixed_digits + $x_count) >= $card_length) {
        return [null, null, null, null, null];
    }

    if ($cvv && (strlen($cvv) < 3 || strlen($cvv) > 4)) {
        return [null, null, null, null, null];
    }

    return [$bin, $month, $year, $cvv, $amount];
}

function format_results($bin, $cards, $amount) {
    $binInfo = get_bin_info(substr(preg_replace('/[xX]/', '', $bin), 0, 6));
    $text = "<b>BIN ⇾</b> $bin\n";
    $text .= "<b>Amount ⇾</b> $amount\n\n";
    foreach ($cards as $card) {
        $text .= "<code>$card</code>\n";
    }
    $text .= "\n";
    if ($binInfo && $binInfo['Status'] === 'SUCCESS') {
        $bank = strtoupper($binInfo['Issuer'] ?? 'Unknown');
        $country_name = $binInfo['Country']['Name'] ?? 'Unknown';
        $country_code = $binInfo['Country']['A2'] ?? '';
        $flag = get_flag($country_code);
        $card_scheme = strtoupper($binInfo['Scheme'] ?? 'Unknown');
        $card_type = strtoupper($binInfo['Type'] ?? 'Unknown');
        $text .= "<b>Bank:</b> $bank\n";
        $text .= "<b>Country:</b> $country_name $flag\n";
        $text .= "<b>BIN Info:</b> $card_scheme - $card_type\n";
    }
    return $text;
}

function handle_gen_command($chat_id, $text, $user_id) {
    save_to_db($chat_id, 'user');
    $loading_msg = bot('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "<b>Generating Credit Cards</b>",
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ]);
    $message_id = json_decode($loading_msg, true)['result']['message_id'];
    $user_input = preg_replace('/^[\/.!#,]gen\s+/i', '', $text);
    list($bin, $month, $year, $cvv, $amount) = parse_input($user_input);
    if (!$bin) {
        bot('editMessageText', [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => "<b>Sorry, BIN must be 6-15 digits❌</b>",
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ]);
        return;
    }
    if ($cvv && (strlen($cvv) < 3 || strlen($cvv) > 4)) {
        bot('editMessageText', [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => "<b>Invalid CVV format. CVV must be 3 or 4 digits ❌</b>",
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ]);
        return;
    }
    if ($amount > CC_GEN_LIMIT) {
        bot('editMessageText', [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => "<b>You can only generate up to 10 credit cards ❌</b>",
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ]);
        return;
    }
    $bin_info = get_bin_info(substr(preg_replace('/[xX]/', '', $bin), 0, 6));
    if (!$bin_info || $bin_info['Status'] !== 'SUCCESS' || !isset($bin_info['Country'])) {
        bot('editMessageText', [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => "<b>❌ Sorry, Invalid BIN Provided</b>",
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ]);
        return;
    }
    usleep(2000000);
    $cards = generate_credit_card($bin, $amount, $month, $year, $cvv);
    if (empty($cards)) {
        bot('editMessageText', [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => "<b>❌ Invalid BIN length for card type</b>",
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ]);
        return;
    }
    $callback_data = "regenerate|$user_input";
    $keyboard = [
        'inline_keyboard' => [[[
            'text' => 'Re-Generate',
            'callback_data' => str_replace(' ', '_', $callback_data)
        ]]]
    ];
    bot('editMessageText', [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => format_results($bin, $cards, $amount),
        'reply_markup' => json_encode($keyboard),
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ]);
}
?>